% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pairwiseWilcox.R
\name{pairwiseWilcox}
\alias{pairwiseWilcox}
\alias{pairwiseWilcox,ANY-method}
\alias{pairwiseWilcox,SummarizedExperiment-method}
\alias{pairwiseWilcox,SingleCellExperiment-method}
\title{Perform pairwise Wilcoxon rank sum tests}
\usage{
pairwiseWilcox(x, ...)

\S4method{pairwiseWilcox}{ANY}(
  x,
  groups,
  block = NULL,
  restrict = NULL,
  exclude = NULL,
  direction = c("any", "up", "down"),
  lfc = 0,
  log.p = FALSE,
  gene.names = NULL,
  subset.row = NULL,
  BPPARAM = SerialParam()
)

\S4method{pairwiseWilcox}{SummarizedExperiment}(x, ..., assay.type = "logcounts")

\S4method{pairwiseWilcox}{SingleCellExperiment}(x, groups = colLabels(x, onAbsence = "error"), ...)
}
\arguments{
\item{x}{A numeric matrix-like object of normalized (and possibly log-transformed) expression values, 
where each column corresponds to a cell and each row corresponds to an endogenous gene.

Alternatively, a \linkS4class{SummarizedExperiment} or \linkS4class{SingleCellExperiment} object containing such a matrix.}

\item{...}{For the generic, further arguments to pass to specific methods.

For the SummarizedExperiment method, further arguments to pass to the ANY method.

For the SingleCellExperiment method, further arguments to pass to the SummarizedExperiment method.}

\item{groups}{A vector of length equal to \code{ncol(x)}, specifying the group assignment for each cell.
If \code{x} is a SingleCellExperiment, this is automatically derived from \code{\link{colLabels}}.}

\item{block}{A factor specifying the blocking level for each cell.}

\item{restrict}{A vector specifying the levels of \code{groups} for which to perform pairwise comparisons.}

\item{exclude}{A vector specifying the levels of \code{groups} for which \emph{not} to perform pairwise comparisons.}

\item{direction}{A string specifying the direction of differences to be considered in the alternative hypothesis.}

\item{lfc}{Numeric scalar specifying the minimum log-fold change for one observation to be considered to be \dQuote{greater} than another.}

\item{log.p}{A logical scalar indicating if log-transformed p-values/FDRs should be returned.}

\item{gene.names}{Deprecated, use \code{row.data} in \code{\link{findMarkers}} instead to add custom annotation.}

\item{subset.row}{See \code{?"\link{scran-gene-selection}"}.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object indicating whether and how parallelization should be performed across genes.}

\item{assay.type}{A string specifying which assay values to use, usually \code{"logcounts"}.}
}
\value{
A list is returned containing \code{statistics} and \code{pairs}.

The \code{statistics} element is itself a list of \linkS4class{DataFrame}s.
Each DataFrame contains the statistics for a comparison between a pair of groups,
including the AUCs, p-values and false discovery rates.

The \code{pairs} element is a DataFrame with one row corresponding to each entry of \code{statistics}.
This contains the fields \code{first} and \code{second}, 
specifying the two groups under comparison in the corresponding DataFrame in \code{statistics}.

In each DataFrame in \code{statistics}, the AUC represents the probability of sampling a value in the \code{first} group greater than a random value from the \code{second} group.
}
\description{
Perform pairwise Wilcoxon rank sum tests between groups of cells, possibly after blocking on uninteresting factors of variation.
}
\details{
This function performs Wilcoxon rank sum tests to identify differentially expressed genes (DEGs) between pairs of groups of cells.
A list of tables is returned where each table contains the statistics for all genes for a comparison between each pair of groups.
This can be examined directly or used as input to \code{\link{combineMarkers}} for marker gene detection.

The effect size for each gene in each comparison is reported as the area under the curve (AUC).
Consider the distribution of expression values for gene X within each of two groups A and B.
The AUC is the probability that a randomly selected cell in A has a greater expression of X than a randomly selected cell in B. 
(Ties are assumed to be randomly broken.)
Concordance probabilities near 0 indicate that most observations in A are lower than most observations in B;
conversely, probabilities near 1 indicate that most observations in A are higher than those in B.
The Wilcoxon rank sum test effectively tests for significant deviations from an AUC of 0.5.

Wilcoxon rank sum tests are more robust to outliers and insensitive to non-normality, in contrast to t-tests in \code{\link{pairwiseTTests}}.
However, they take longer to run, the effect sizes are less interpretable, and there are more subtle violations of its assumptions in real data.
For example, the i.i.d. assumptions are unlikely to hold after scaling normalization due to differences in variance.
Also note that we approximate the distribution of the Wilcoxon rank sum statistic to deal with large numbers of cells and ties.

If \code{restrict} is specified, comparisons are only performed between pairs of groups in \code{restrict}.
This can be used to focus on DEGs distinguishing between a subset of the groups (e.g., closely related cell subtypes).

If \code{exclude} is specified, comparisons are not performed between groups in \code{exclude}.
Similarly, if any entries of \code{groups} are \code{NA}, the corresponding cells are are ignored.
}
\section{Direction and magnitude of the effect}{

If \code{direction="any"}, two-sided Wilcoxon rank sum tests will be performed for each pairwise comparisons between groups of cells.
Otherwise, one-sided tests in the specified direction will be used instead.
This can be used to focus on genes that are upregulated in each group of interest, which is often easier to interpret.

To interpret the setting of \code{direction}, consider the DataFrame for group X, in which we are comparing to another group Y.
If \code{direction="up"}, genes will only be significant in this DataFrame if they are upregulated in group X compared to Y.
If \code{direction="down"}, genes will only be significant if they are downregulated in group X compared to Y.
See \code{?\link{wilcox.test}} for more details on the interpretation of one-sided Wilcoxon rank sum tests.

Users can also specify a log-fold change threshold in \code{lfc} to focus on genes that exhibit large shifts in location.
This is equivalent to specifying the \code{mu} parameter in \code{\link{wilcox.test}}, 
with some additional subtleties depending on \code{direction}:
\itemize{
\item If \code{direction="any"}, the null hypothesis is that the true shift is either \code{-lfc} or \code{lfc} with equal probability.
A two-sided p-value is computed against this composite null.
The AUC is computed by averaging the AUCs obtained when X's expression values are shifted to the left or right by \code{lfc}.
This can be very roughly interpreted as considering an observation of Y to be tied with an observation of X if they differ by less than \code{lfc}.
\item If \code{direction="up"}, the null hypothesis is that the true shift is \code{lfc}, and a one-sided p-value is computed.
The AUC is computed after shifting X's expression values to the left by \code{lfc}.
\item If \code{direction="down"}, the null hypothesis is that the true shift is \code{-lfc}, and a one-sided p-value is computed.
The AUC is computed after shifting X's expression values to the right by \code{lfc}.
}
The fact that the AUC depends on \code{lfc} is necessary to preserve its relationship with the p-value.
}

\section{Blocking on uninteresting factors}{

If \code{block} is specified, Wilcoxon tests are performed between groups of cells within each level of \code{block}.
For each pair of groups, the p-values for each gene across all levels of \code{block} are combined using Stouffer's Z-score method.
The reported AUC is also a weighted average of the AUCs across all levels.

The weight for a particular level of \code{block} is defined as \eqn{N_xN_y},
where \eqn{N_x} and \eqn{N_y} are the number of cells in groups X and Y, respectively, for that level. 
This means that p-values from blocks with more cells will have a greater contribution to the combined p-value for each gene.

When combining across batches, one-sided p-values in the same direction are combined first.
Then, if \code{direction="any"}, the two combined p-values from both directions are combined.
This ensures that a gene only receives a low overall p-value if it changes in the same direction across batches.

When comparing two groups, blocking levels are ignored if no p-value was reported, e.g., if there were insufficient cells for a group in a particular level. 
If all levels are ignored in this manner, the entire comparison will only contain \code{NA} p-values and a warning will be emitted.
}

\examples{
library(scuttle)
sce <- mockSCE()
sce <- logNormCounts(sce)

# Any clustering method is okay.
kout <- kmeans(t(logcounts(sce)), centers=2) 

# Vanilla application:
out <- pairwiseWilcox(logcounts(sce), groups=kout$cluster)
out

# Directional and with a minimum log-fold change:
out <- pairwiseWilcox(logcounts(sce), groups=kout$cluster, 
    direction="up", lfc=0.2)
out

}
\references{
Whitlock MC (2005). 
Combining probability from independent tests: the weighted Z-method is superior to Fisher's approach. 
\emph{J. Evol. Biol.} 18, 5:1368-73.

Soneson C and Robinson MD (2018). 
Bias, robustness and scalability in single-cell differential expression analysis. 
\emph{Nat. Methods}
}
\seealso{
\code{\link{wilcox.test}}, on which this function is based.

\code{\link{combineMarkers}}, to combine pairwise comparisons into a single DataFrame per group.

\code{\link{getTopMarkers}}, to obtain the top markers from each pairwise comparison.
}
\author{
Aaron Lun
}
