/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Hidayat Khan <huk2209@gmail.com>
 */

#define SIMDE_TEST_X86_AVX512_INSN srl

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/srl.h>

static int
test_simde_mm512_srl_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t a[32];
    const int64_t b[2];
    const int16_t r[32];
  } test_vec[] = {
    { { -INT16_C( 21752), -INT16_C( 22194), -INT16_C( 18737),  INT16_C(  8433),  INT16_C( 23052),  INT16_C( 22961), -INT16_C(  7722), -INT16_C( 29326),
         INT16_C(  6474),  INT16_C( 15054),  INT16_C(  9100), -INT16_C( 16434), -INT16_C( 23329),  INT16_C( 14938), -INT16_C( 23326), -INT16_C(  5412),
         INT16_C( 10831),  INT16_C(  8083), -INT16_C( 31264), -INT16_C(  4801), -INT16_C(  3873), -INT16_C( 18874), -INT16_C( 18223),  INT16_C(  7235),
         INT16_C(  4562),  INT16_C( 24150),  INT16_C(  9268),  INT16_C(  4893),  INT16_C( 30920), -INT16_C( 21939),  INT16_C( 10524),  INT16_C( 27541) },
      {  INT64_C(                  17),  INT64_C(                  30) },
      {  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { { -INT16_C(   616), -INT16_C( 13232),  INT16_C( 28193), -INT16_C(  5664),  INT16_C( 11750),  INT16_C(   660),  INT16_C( 10583), -INT16_C( 21907),
        -INT16_C(  1967), -INT16_C(    34), -INT16_C(    63),  INT16_C( 31628), -INT16_C( 12441), -INT16_C( 30970), -INT16_C( 21163), -INT16_C(  4743),
        -INT16_C( 13910), -INT16_C( 13382), -INT16_C( 26057),  INT16_C(  7604),  INT16_C( 18631),  INT16_C(  7711), -INT16_C( 29327), -INT16_C( 15415),
        -INT16_C( 22651),  INT16_C( 18114),  INT16_C( 20135),  INT16_C(  3777), -INT16_C( 14563),  INT16_C( 29333),  INT16_C(  3700),  INT16_C(  7776) },
      {  INT64_C(                  15),  INT64_C(                  11) },
      {  INT16_C(     1),  INT16_C(     1),  INT16_C(     0),  INT16_C(     1),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     1),
         INT16_C(     1),  INT16_C(     1),  INT16_C(     1),  INT16_C(     0),  INT16_C(     1),  INT16_C(     1),  INT16_C(     1),  INT16_C(     1),
         INT16_C(     1),  INT16_C(     1),  INT16_C(     1),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     1),  INT16_C(     1),
         INT16_C(     1),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     1),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { { -INT16_C(  8950), -INT16_C( 20060),  INT16_C( 26155),  INT16_C( 18623),  INT16_C( 21549), -INT16_C( 23878),  INT16_C(  6754),  INT16_C( 15040),
        -INT16_C( 21964), -INT16_C(  6071),  INT16_C( 30024),  INT16_C( 11876), -INT16_C(   319), -INT16_C( 25978), -INT16_C( 24223),  INT16_C( 27640),
        -INT16_C( 25218), -INT16_C( 22243), -INT16_C(  9213),  INT16_C( 12529), -INT16_C( 21455), -INT16_C( 27694), -INT16_C( 27706), -INT16_C(  1075),
         INT16_C(  5693), -INT16_C( 31261),  INT16_C( 18316),  INT16_C( 19891),  INT16_C( 14917), -INT16_C( 22808), -INT16_C(  7973),  INT16_C( 23058) },
      {  INT64_C(                   7),  INT64_C(                  10) },
      {  INT16_C(   442),  INT16_C(   355),  INT16_C(   204),  INT16_C(   145),  INT16_C(   168),  INT16_C(   325),  INT16_C(    52),  INT16_C(   117),
         INT16_C(   340),  INT16_C(   464),  INT16_C(   234),  INT16_C(    92),  INT16_C(   509),  INT16_C(   309),  INT16_C(   322),  INT16_C(   215),
         INT16_C(   314),  INT16_C(   338),  INT16_C(   440),  INT16_C(    97),  INT16_C(   344),  INT16_C(   295),  INT16_C(   295),  INT16_C(   503),
         INT16_C(    44),  INT16_C(   267),  INT16_C(   143),  INT16_C(   155),  INT16_C(   116),  INT16_C(   333),  INT16_C(   449),  INT16_C(   180) } },
    { {  INT16_C( 18100),  INT16_C( 16600), -INT16_C( 29555), -INT16_C( 11379),  INT16_C( 30150), -INT16_C( 24199), -INT16_C( 29866), -INT16_C( 11269),
        -INT16_C(    70), -INT16_C( 14764),  INT16_C(  1524), -INT16_C( 27390), -INT16_C( 11640), -INT16_C( 24580),  INT16_C( 24432),  INT16_C(  9458),
        -INT16_C( 13403),  INT16_C( 12900), -INT16_C(  3753),  INT16_C(  7429),  INT16_C( 32615), -INT16_C( 16962), -INT16_C( 17910), -INT16_C( 14960),
        -INT16_C(  6983), -INT16_C( 21109), -INT16_C( 29207),  INT16_C( 29250),  INT16_C( 15968), -INT16_C( 12271),  INT16_C(   925),  INT16_C( 17140) },
      {  INT64_C(                  22),  INT64_C(                  14) },
      {  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { {  INT16_C( 21730), -INT16_C( 13279),  INT16_C( 25569),  INT16_C( 16702),  INT16_C( 20385),  INT16_C( 16145),  INT16_C(  1362),  INT16_C(  8577),
        -INT16_C(  2467), -INT16_C( 22714), -INT16_C( 30351),  INT16_C( 27223), -INT16_C( 14966),  INT16_C( 17774),  INT16_C( 14275), -INT16_C( 23111),
        -INT16_C(  9589),  INT16_C( 28017), -INT16_C( 20675), -INT16_C(  8530), -INT16_C( 16130),  INT16_C( 20765), -INT16_C( 24635),  INT16_C(  9074),
        -INT16_C( 18283),  INT16_C(  1738),  INT16_C(  8513), -INT16_C( 13455), -INT16_C(  8218), -INT16_C( 22256), -INT16_C( 14057), -INT16_C( 23985) },
      {  INT64_C(                  13),  INT64_C(                   2) },
      {  INT16_C(     2),  INT16_C(     6),  INT16_C(     3),  INT16_C(     2),  INT16_C(     2),  INT16_C(     1),  INT16_C(     0),  INT16_C(     1),
         INT16_C(     7),  INT16_C(     5),  INT16_C(     4),  INT16_C(     3),  INT16_C(     6),  INT16_C(     2),  INT16_C(     1),  INT16_C(     5),
         INT16_C(     6),  INT16_C(     3),  INT16_C(     5),  INT16_C(     6),  INT16_C(     6),  INT16_C(     2),  INT16_C(     4),  INT16_C(     1),
         INT16_C(     5),  INT16_C(     0),  INT16_C(     1),  INT16_C(     6),  INT16_C(     6),  INT16_C(     5),  INT16_C(     6),  INT16_C(     5) } },
    { {  INT16_C( 12522),  INT16_C( 11031), -INT16_C( 30638),  INT16_C( 14583),  INT16_C(  1896),  INT16_C( 32738),  INT16_C( 12753),  INT16_C( 29729),
         INT16_C( 12785),  INT16_C( 24917),  INT16_C(  5359),  INT16_C( 28112), -INT16_C( 28688),  INT16_C( 27824),  INT16_C(  6081), -INT16_C( 21635),
        -INT16_C( 27577), -INT16_C( 26154), -INT16_C( 13027), -INT16_C( 31278), -INT16_C( 19243), -INT16_C( 23036),  INT16_C(  9701), -INT16_C( 10726),
         INT16_C( 28502),  INT16_C( 17720),  INT16_C(  2179),  INT16_C( 29874),  INT16_C( 25495),  INT16_C( 22752),  INT16_C( 23930), -INT16_C( 16125) },
      {  INT64_C(                  15),  INT64_C(                   2) },
      {  INT16_C(     0),  INT16_C(     0),  INT16_C(     1),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     1),  INT16_C(     0),  INT16_C(     0),  INT16_C(     1),
         INT16_C(     1),  INT16_C(     1),  INT16_C(     1),  INT16_C(     1),  INT16_C(     1),  INT16_C(     1),  INT16_C(     0),  INT16_C(     1),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     1) } },
    { { -INT16_C( 11092),  INT16_C( 12377),  INT16_C(  3036),  INT16_C( 29860), -INT16_C( 31634), -INT16_C(  5940), -INT16_C( 12063), -INT16_C( 11606),
         INT16_C(  1450),  INT16_C( 20961),  INT16_C( 29746),  INT16_C(  5070), -INT16_C(  4084), -INT16_C( 13863),  INT16_C( 29997), -INT16_C(  9508),
         INT16_C( 13642),  INT16_C(  9738), -INT16_C( 20927), -INT16_C( 20582),  INT16_C( 26418),  INT16_C(  5016),  INT16_C( 16951), -INT16_C(  7707),
        -INT16_C( 14777),  INT16_C( 31026),  INT16_C(    59),  INT16_C( 18316),  INT16_C( 26097),  INT16_C(  7696), -INT16_C(  4902),  INT16_C(  9464) },
      {  INT64_C(                  28),  INT64_C(                   6) },
      {  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { { -INT16_C( 26719), -INT16_C(  9044),  INT16_C( 14487), -INT16_C( 30685),  INT16_C( 13213),  INT16_C( 30887), -INT16_C( 24800),  INT16_C( 17052),
        -INT16_C(  6238),  INT16_C( 21157), -INT16_C( 18483),  INT16_C(  6453),  INT16_C( 10850),  INT16_C( 20125),  INT16_C(   261), -INT16_C( 22654),
         INT16_C( 11928),  INT16_C( 12419), -INT16_C( 22681),  INT16_C(  1208),  INT16_C( 24538), -INT16_C(  1412),  INT16_C(  6655), -INT16_C( 24260),
        -INT16_C(  7936), -INT16_C( 12813),  INT16_C( 10393), -INT16_C(  1049), -INT16_C( 31661),  INT16_C( 22601), -INT16_C( 13435),  INT16_C(  7935) },
      {  INT64_C(                   0),  INT64_C(                  31) },
      { -INT16_C( 26719), -INT16_C(  9044),  INT16_C( 14487), -INT16_C( 30685),  INT16_C( 13213),  INT16_C( 30887), -INT16_C( 24800),  INT16_C( 17052),
        -INT16_C(  6238),  INT16_C( 21157), -INT16_C( 18483),  INT16_C(  6453),  INT16_C( 10850),  INT16_C( 20125),  INT16_C(   261), -INT16_C( 22654),
         INT16_C( 11928),  INT16_C( 12419), -INT16_C( 22681),  INT16_C(  1208),  INT16_C( 24538), -INT16_C(  1412),  INT16_C(  6655), -INT16_C( 24260),
        -INT16_C(  7936), -INT16_C( 12813),  INT16_C( 10393), -INT16_C(  1049), -INT16_C( 31661),  INT16_C( 22601), -INT16_C( 13435),  INT16_C(  7935) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_srl_epi16(a, b);
    simde_test_x86_assert_equal_i16x32(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_srl_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[16];
    const int64_t b[2];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(  1475230993),  INT32_C(   227741951), -INT32_C(  1586039920), -INT32_C(   909665363),  INT32_C(  1410969055),  INT32_C(   106491510),  INT32_C(   829046348),  INT32_C(  1384343139),
         INT32_C(   670733352),  INT32_C(   892636837),  INT32_C(   785820289),  INT32_C(   620272965), -INT32_C(  1049030325),  INT32_C(  1288163584),  INT32_C(  1920807183), -INT32_C(  1060831080) },
      {  INT64_C(                  25),  INT64_C(                   5) },
      {  INT32_C(          84),  INT32_C(           6),  INT32_C(          80),  INT32_C(         100),  INT32_C(          42),  INT32_C(           3),  INT32_C(          24),  INT32_C(          41),
         INT32_C(          19),  INT32_C(          26),  INT32_C(          23),  INT32_C(          18),  INT32_C(          96),  INT32_C(          38),  INT32_C(          57),  INT32_C(          96) } },
    { {  INT32_C(   116369926), -INT32_C(  1974290821), -INT32_C(  2130849816),  INT32_C(  1681965775), -INT32_C(   811784063),  INT32_C(   211618885),  INT32_C(   857250385), -INT32_C(  1772009073),
         INT32_C(  1855738099), -INT32_C(   285610234), -INT32_C(  1905330498),  INT32_C(   972206008),  INT32_C(   470323159), -INT32_C(   332880229), -INT32_C(   803192512),  INT32_C(  2120647050) },
      {  INT64_C(                  22),  INT64_C(                  23) },
      {  INT32_C(          27),  INT32_C(         553),  INT32_C(         515),  INT32_C(         401),  INT32_C(         830),  INT32_C(          50),  INT32_C(         204),  INT32_C(         601),
         INT32_C(         442),  INT32_C(         955),  INT32_C(         569),  INT32_C(         231),  INT32_C(         112),  INT32_C(         944),  INT32_C(         832),  INT32_C(         505) } },
    { {  INT32_C(  1540871615), -INT32_C(  1136197509), -INT32_C(   879990975), -INT32_C(  1186336024), -INT32_C(   426690828), -INT32_C(   141142245),  INT32_C(  1871369355), -INT32_C(  1020635388),
        -INT32_C(  1474428627),  INT32_C(  1113875969), -INT32_C(  1240534835), -INT32_C(   680568862), -INT32_C(  1447231347), -INT32_C(   476032169),  INT32_C(   726805031), -INT32_C(  1343325054) },
      {  INT64_C(                   8),  INT64_C(                  22) },
      {  INT32_C(     6019029),  INT32_C(    12338944),  INT32_C(    13339751),  INT32_C(    12143090),  INT32_C(    15110454),  INT32_C(    16225879),  INT32_C(     7310036),  INT32_C(    12790359),
         INT32_C(    11017729),  INT32_C(     4351078),  INT32_C(    11931376),  INT32_C(    14118743),  INT32_C(    11123968),  INT32_C(    14917715),  INT32_C(     2839082),  INT32_C(    11529852) } },
    { { -INT32_C(  1101192346), -INT32_C(  1650328202), -INT32_C(  1429671128), -INT32_C(   329664913),  INT32_C(   913027267),  INT32_C(   427240812),  INT32_C(   598240764),  INT32_C(   987238099),
        -INT32_C(  1476905679),  INT32_C(  1531222323), -INT32_C(    83555188), -INT32_C(  2031657278),  INT32_C(  2075939598),  INT32_C(  2090087296),  INT32_C(  1956658337),  INT32_C(  2075031626) },
      {  INT64_C(                  25),  INT64_C(                  21) },
      {  INT32_C(          95),  INT32_C(          78),  INT32_C(          85),  INT32_C(         118),  INT32_C(          27),  INT32_C(          12),  INT32_C(          17),  INT32_C(          29),
         INT32_C(          83),  INT32_C(          45),  INT32_C(         125),  INT32_C(          67),  INT32_C(          61),  INT32_C(          62),  INT32_C(          58),  INT32_C(          61) } },
    { { -INT32_C(  2128054527),  INT32_C(  1274920106),  INT32_C(  1119919608), -INT32_C(  1010995691),  INT32_C(  1420025621), -INT32_C(  1222582459), -INT32_C(  1091836385), -INT32_C(  1788107116),
        -INT32_C(   954821859),  INT32_C(  1192432719), -INT32_C(   947268687),  INT32_C(  1451902529),  INT32_C(  1789603109),  INT32_C(   740412172),  INT32_C(  1189743793), -INT32_C(   807709262) },
      {  INT64_C(                  22),  INT64_C(                  14) },
      {  INT32_C(         516),  INT32_C(         303),  INT32_C(         267),  INT32_C(         782),  INT32_C(         338),  INT32_C(         732),  INT32_C(         763),  INT32_C(         597),
         INT32_C(         796),  INT32_C(         284),  INT32_C(         798),  INT32_C(         346),  INT32_C(         426),  INT32_C(         176),  INT32_C(         283),  INT32_C(         831) } },
    { {  INT32_C(  1121763382),  INT32_C(  1013906827), -INT32_C(  1132308471), -INT32_C(  1786028371),  INT32_C(  1456218704),  INT32_C(  1225607884), -INT32_C(  1643606959), -INT32_C(   904913516),
         INT32_C(  1745743069), -INT32_C(   207324183), -INT32_C(  2119227436), -INT32_C(   719897979),  INT32_C(   690742109),  INT32_C(  2138257454),  INT32_C(  1495169988),  INT32_C(  1965239960) },
      {  INT64_C(                  16),  INT64_C(                   7) },
      {  INT32_C(       17116),  INT32_C(       15470),  INT32_C(       48258),  INT32_C(       38283),  INT32_C(       22220),  INT32_C(       18701),  INT32_C(       40456),  INT32_C(       51728),
         INT32_C(       26637),  INT32_C(       62372),  INT32_C(       33199),  INT32_C(       54551),  INT32_C(       10539),  INT32_C(       32627),  INT32_C(       22814),  INT32_C(       29987) } },
    { {  INT32_C(   711405052),  INT32_C(   715774566), -INT32_C(   310130859),  INT32_C(   291678198), -INT32_C(  2095759401),  INT32_C(  1761807809), -INT32_C(  1802041933),  INT32_C(   433232157),
         INT32_C(   759380679),  INT32_C(  1784147220), -INT32_C(  1437082700),  INT32_C(  1505475202), -INT32_C(  1159867911), -INT32_C(  1859854114),  INT32_C(    52870117),  INT32_C(   454883412) },
      {  INT64_C(                   0),  INT64_C(                  23) },
      {  INT32_C(   711405052),  INT32_C(   715774566), -INT32_C(   310130859),  INT32_C(   291678198), -INT32_C(  2095759401),  INT32_C(  1761807809), -INT32_C(  1802041933),  INT32_C(   433232157),
         INT32_C(   759380679),  INT32_C(  1784147220), -INT32_C(  1437082700),  INT32_C(  1505475202), -INT32_C(  1159867911), -INT32_C(  1859854114),  INT32_C(    52870117),  INT32_C(   454883412) } },
    { {  INT32_C(   376845112), -INT32_C(   106391020), -INT32_C(  1426272683), -INT32_C(   104523322), -INT32_C(   968880519),  INT32_C(   700969390), -INT32_C(  1138330631),  INT32_C(   326663387),
        -INT32_C(  1003819344), -INT32_C(   557985143),  INT32_C(  1720236704),  INT32_C(  1281314515),  INT32_C(   168992604), -INT32_C(  1976313456),  INT32_C(   675699021), -INT32_C(  2059682091) },
      {  INT64_C(                  26),  INT64_C(                   8) },
      {  INT32_C(           5),  INT32_C(          62),  INT32_C(          42),  INT32_C(          62),  INT32_C(          49),  INT32_C(          10),  INT32_C(          47),  INT32_C(           4),
         INT32_C(          49),  INT32_C(          55),  INT32_C(          25),  INT32_C(          19),  INT32_C(           2),  INT32_C(          34),  INT32_C(          10),  INT32_C(          33) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_srl_epi32(a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_srl_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a[16];
    const int64_t b[2];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(  1391069190),  INT32_C(    76515171), -INT32_C(  1669093777),  INT32_C(  2052374992), -INT32_C(    17678478), -INT32_C(  1259542507),  INT32_C(   626340292),  INT32_C(  1967227002),
        -INT32_C(   618506633),  INT32_C(  1306505694),  INT32_C(   669606487), -INT32_C(  1784595165), -INT32_C(  1835822212),  INT32_C(  1078361979),  INT32_C(   728079281), -INT32_C(  1818188005) },
      UINT16_C(49917),
      {  INT32_C(  1299438446),  INT32_C(   313510440), -INT32_C(  1840262415), -INT32_C(    98186137), -INT32_C(  1535467170),  INT32_C(  1178545121), -INT32_C(   135439530),  INT32_C(  1555688430),
        -INT32_C(   274125370), -INT32_C(   385787657), -INT32_C(  1854189526),  INT32_C(  2055971100),  INT32_C(   589169986), -INT32_C(  2023137744), -INT32_C(  1484892231),  INT32_C(   134428482) },
      {  INT64_C(                   5),  INT64_C(                  20) },
      {  INT32_C(    40607451),  INT32_C(    76515171),  INT32_C(    76709527),  INT32_C(   131149411),  INT32_C(    86234378),  INT32_C(    36829535),  INT32_C(   129985242),  INT32_C(    48615263),
        -INT32_C(   618506633),  INT32_C(   122161863),  INT32_C(   669606487), -INT32_C(  1784595165), -INT32_C(  1835822212),  INT32_C(  1078361979),  INT32_C(    87814845),  INT32_C(     4200890) } },
    { { -INT32_C(  2134442416),  INT32_C(   352792668), -INT32_C(   910391929),  INT32_C(   567394237),  INT32_C(  1870514539),  INT32_C(   178111169), -INT32_C(   462398333), -INT32_C(   142061401),
        -INT32_C(  1418244530),  INT32_C(    96501630), -INT32_C(  1043432188), -INT32_C(  1461477316),  INT32_C(   706175081), -INT32_C(  1506429661), -INT32_C(  1198873327),  INT32_C(  1135547125) },
      UINT16_C(10081),
      { -INT32_C(  1348018194), -INT32_C(  1288918299),  INT32_C(  1330866284),  INT32_C(   682605584),  INT32_C(   484299495),  INT32_C(    63041144),  INT32_C(  1444263591), -INT32_C(   394430727),
         INT32_C(   999760726),  INT32_C(   988726222),  INT32_C(  1015628331), -INT32_C(   446416642), -INT32_C(  2097069813), -INT32_C(   695811537),  INT32_C(  1915591800),  INT32_C(  1750772242) },
      {  INT64_C(                  27),  INT64_C(                  20) },
      {  INT32_C(          21),  INT32_C(   352792668), -INT32_C(   910391929),  INT32_C(   567394237),  INT32_C(  1870514539),  INT32_C(           0),  INT32_C(          10), -INT32_C(   142061401),
         INT32_C(           7),  INT32_C(           7),  INT32_C(           7), -INT32_C(  1461477316),  INT32_C(   706175081),  INT32_C(          26), -INT32_C(  1198873327),  INT32_C(  1135547125) } },
    { { -INT32_C(   265897536), -INT32_C(   188306308),  INT32_C(  1533473608),  INT32_C(  1824768158),  INT32_C(  1695049649), -INT32_C(   851058951), -INT32_C(   727752643), -INT32_C(  1602791456),
        -INT32_C(  1919902447), -INT32_C(  1837017271), -INT32_C(   387061686),  INT32_C(  1515499688),  INT32_C(   280976407),  INT32_C(  2010973242),  INT32_C(  1162640741), -INT32_C(   437926956) },
      UINT16_C(30304),
      { -INT32_C(   187848334),  INT32_C(   685512507),  INT32_C(  1423541248), -INT32_C(  1632505634), -INT32_C(   559748351), -INT32_C(  1352988829),  INT32_C(   846344268), -INT32_C(  2002202091),
        -INT32_C(  1216580229), -INT32_C(  1931519860), -INT32_C(  1142834980), -INT32_C(  1436970327), -INT32_C(   527893635),  INT32_C(  1334830083),  INT32_C(   696320276),  INT32_C(  1337010643) },
      {  INT64_C(                  12),  INT64_C(                  21) },
      { -INT32_C(   265897536), -INT32_C(   188306308),  INT32_C(  1533473608),  INT32_C(  1824768158),  INT32_C(  1695049649),  INT32_C(      718256),  INT32_C(      206627), -INT32_C(  1602791456),
        -INT32_C(  1919902447),  INT32_C(      577013),  INT32_C(      769563),  INT32_C(  1515499688),  INT32_C(      919695),  INT32_C(      325886),  INT32_C(      170000), -INT32_C(   437926956) } },
    { {  INT32_C(  1955101041),  INT32_C(  1908676701), -INT32_C(   308591335),  INT32_C(   222055535), -INT32_C(    13090182), -INT32_C(   983437273), -INT32_C(  1291026808),  INT32_C(  1717304820),
        -INT32_C(   723852425),  INT32_C(   356949755),  INT32_C(  1392697828), -INT32_C(  1486864851), -INT32_C(  1482188416),  INT32_C(   292358281),  INT32_C(  2076473735), -INT32_C(   287236233) },
      UINT16_C(48131),
      {  INT32_C(   140181186),  INT32_C(   367607315), -INT32_C(   229435503),  INT32_C(  1703662526), -INT32_C(   395438981), -INT32_C(   362679003),  INT32_C(  1376376944), -INT32_C(  1995567930),
         INT32_C(   512845835),  INT32_C(   942963623),  INT32_C(  1344964498), -INT32_C(   692669093), -INT32_C(   272751415), -INT32_C(  1982259431), -INT32_C(  1092885768), -INT32_C(    95950353) },
      {  INT64_C(                  28),  INT64_C(                  27) },
      {  INT32_C(           0),  INT32_C(           1), -INT32_C(   308591335),  INT32_C(   222055535), -INT32_C(    13090182), -INT32_C(   983437273), -INT32_C(  1291026808),  INT32_C(  1717304820),
        -INT32_C(   723852425),  INT32_C(   356949755),  INT32_C(           5),  INT32_C(          13),  INT32_C(          14),  INT32_C(           8),  INT32_C(  2076473735),  INT32_C(          15) } },
    { {  INT32_C(   701154064), -INT32_C(   625761310),  INT32_C(  1956220549), -INT32_C(   898703240),  INT32_C(   230918073), -INT32_C(  1477184301),  INT32_C(  1658202704),  INT32_C(   658365206),
        -INT32_C(  2125461602),  INT32_C(   794493866),  INT32_C(   178582674), -INT32_C(  1898704171),  INT32_C(  1838978969), -INT32_C(   602632309), -INT32_C(   801182791),  INT32_C(  1710717894) },
      UINT16_C(18306),
      {  INT32_C(  1095445734), -INT32_C(    13181605), -INT32_C(  1156445209),  INT32_C(   894610329),  INT32_C(   767941912), -INT32_C(   149389639),  INT32_C(  1165155918),  INT32_C(   697168963),
         INT32_C(  2087442464), -INT32_C(  1669619275),  INT32_C(  1163365804),  INT32_C(  1367058745),  INT32_C(  1082015878),  INT32_C(   221746878), -INT32_C(  1202541963), -INT32_C(  1075650658) },
      {  INT64_C(                  19),  INT64_C(                  17) },
      {  INT32_C(   701154064),  INT32_C(        8166),  INT32_C(  1956220549), -INT32_C(   898703240),  INT32_C(   230918073), -INT32_C(  1477184301),  INT32_C(  1658202704),  INT32_C(        1329),
         INT32_C(        3981),  INT32_C(        5007),  INT32_C(        2218), -INT32_C(  1898704171),  INT32_C(  1838978969), -INT32_C(   602632309),  INT32_C(        5898),  INT32_C(  1710717894) } },
    { {  INT32_C(  1506758042),  INT32_C(  1483081443),  INT32_C(  1326561456),  INT32_C(  1326379928),  INT32_C(   784091456),  INT32_C(  1137231103),  INT32_C(   750823204), -INT32_C(  1665429758),
        -INT32_C(  1074427172),  INT32_C(  1092115345), -INT32_C(  1399838444),  INT32_C(  1559993884), -INT32_C(   410339353), -INT32_C(  1607839108), -INT32_C(    70456327),  INT32_C(  1452836986) },
      UINT16_C(36114),
      {  INT32_C(   770286357),  INT32_C(  1968635365), -INT32_C(  1542163799),  INT32_C(  1482488782),  INT32_C(   229300450),  INT32_C(  1157145720),  INT32_C(   936145567), -INT32_C(   574234680),
         INT32_C(  1728818818),  INT32_C(  1423794603), -INT32_C(  1560743468), -INT32_C(   805612308), -INT32_C(  1529043668), -INT32_C(   370551735),  INT32_C(   237024582),  INT32_C(   401335700) },
      {  INT64_C(                   5),  INT64_C(                   1) },
      {  INT32_C(  1506758042),  INT32_C(    61519855),  INT32_C(  1326561456),  INT32_C(  1326379928),  INT32_C(     7165639),  INT32_C(  1137231103),  INT32_C(   750823204), -INT32_C(  1665429758),
         INT32_C(    54025588),  INT32_C(  1092115345),  INT32_C(    85444494),  INT32_C(   109042343), -INT32_C(   410339353), -INT32_C(  1607839108), -INT32_C(    70456327),  INT32_C(    12541740) } },
    { { -INT32_C(  1196366737), -INT32_C(    22963784), -INT32_C(   485703089),  INT32_C(  1006303143),  INT32_C(  1182366190),  INT32_C(   561122516),  INT32_C(  1985626263),  INT32_C(  2038587914),
        -INT32_C(    30330042),  INT32_C(   469554124),  INT32_C(  1023346837), -INT32_C(   310904321),  INT32_C(  1194586482), -INT32_C(  1805081091), -INT32_C(   267730202),  INT32_C(  1785302308) },
      UINT16_C(39616),
      {  INT32_C(  1684900968), -INT32_C(  1486093656), -INT32_C(  1231000769), -INT32_C(  1935207591), -INT32_C(  1036868518),  INT32_C(  1132730424),  INT32_C(  1909499912),  INT32_C(  2047578130),
        -INT32_C(   992052964), -INT32_C(  1167373701),  INT32_C(   242289845),  INT32_C(  2040207391), -INT32_C(   180630083), -INT32_C(   315047963),  INT32_C(  2036205671), -INT32_C(  1242338920) },
      {  INT64_C(                  26),  INT64_C(                  11) },
      { -INT32_C(  1196366737), -INT32_C(    22963784), -INT32_C(   485703089),  INT32_C(  1006303143),  INT32_C(  1182366190),  INT32_C(   561122516),  INT32_C(          28),  INT32_C(          30),
        -INT32_C(    30330042),  INT32_C(          46),  INT32_C(  1023346837),  INT32_C(          30),  INT32_C(          61), -INT32_C(  1805081091), -INT32_C(   267730202),  INT32_C(          45) } },
    { {  INT32_C(   743752775), -INT32_C(   367424125),  INT32_C(   778270613), -INT32_C(  1008511264),  INT32_C(  1109482535),  INT32_C(   840055105),  INT32_C(  1698886083), -INT32_C(  1295725717),
         INT32_C(   316545167),  INT32_C(  1006434213),  INT32_C(  1332305774), -INT32_C(   602780491), -INT32_C(   367119448),  INT32_C(   790376812), -INT32_C(  1902878942), -INT32_C(  1170188246) },
      UINT16_C( 8049),
      { -INT32_C(   921299252), -INT32_C(  1171749551), -INT32_C(   435757356), -INT32_C(   652628038), -INT32_C(  1257601639),  INT32_C(  1226058933), -INT32_C(    73252934), -INT32_C(   988146695),
         INT32_C(  2072916009), -INT32_C(  1992968267),  INT32_C(  1332690069), -INT32_C(  2077718293), -INT32_C(  1019661810),  INT32_C(   420236895), -INT32_C(  2045464947), -INT32_C(   347394367) },
      {  INT64_C(                   4),  INT64_C(                  20) },
      {  INT32_C(   210854252), -INT32_C(   367424125),  INT32_C(   778270613), -INT32_C(  1008511264),  INT32_C(   189835353),  INT32_C(    76628683),  INT32_C(   263857147), -INT32_C(  1295725717),
         INT32_C(   129557250),  INT32_C(   143874939),  INT32_C(    83293129),  INT32_C(   138578062),  INT32_C(   204706592),  INT32_C(   790376812), -INT32_C(  1902878942), -INT32_C(  1170188246) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_srl_epi32(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_srl_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const int32_t a[16];
    const int64_t b[2];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(10508),
      { -INT32_C(   652285907),  INT32_C(  1973831532),  INT32_C(  1688648047), -INT32_C(  1906907068),  INT32_C(  2083047916), -INT32_C(  1402768041), -INT32_C(  2074714565), -INT32_C(  1683135890),
        -INT32_C(  1217017014), -INT32_C(  2111038702),  INT32_C(   434557909), -INT32_C(  1482146373),  INT32_C(  1562693638), -INT32_C(  2130081978),  INT32_C(  1392861157),  INT32_C(   217035713) },
      {  INT64_C(                  10),  INT64_C(                   7) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(     1649070),  INT32_C(     2332090),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
         INT32_C(     3005810),  INT32_C(           0),  INT32_C(           0),  INT32_C(     2746895),  INT32_C(           0),  INT32_C(     2114145),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(18918),
      { -INT32_C(  1563415399), -INT32_C(  1291733586), -INT32_C(   144260344), -INT32_C(  1839471153), -INT32_C(  1987978378), -INT32_C(  1652407249), -INT32_C(   424429119), -INT32_C(  1892705547),
         INT32_C(  1949433798), -INT32_C(  1121438796), -INT32_C(   978022666), -INT32_C(   380170125),  INT32_C(   443734251), -INT32_C(   558303972), -INT32_C(  1547408466), -INT32_C(   869075963) },
      {  INT64_C(                  17),  INT64_C(                   2) },
      {  INT32_C(           0),  INT32_C(       22912),  INT32_C(       31667),  INT32_C(           0),  INT32_C(           0),  INT32_C(       20161),  INT32_C(       29529),  INT32_C(       18327),
         INT32_C(       14872),  INT32_C(           0),  INT32_C(           0),  INT32_C(       29867),  INT32_C(           0),  INT32_C(           0),  INT32_C(       20962),  INT32_C(           0) } },
    { UINT16_C(50817),
      { -INT32_C(   407134673),  INT32_C(  1079142780),  INT32_C(  1060395021),  INT32_C(  1688414244),  INT32_C(   902642384),  INT32_C(   424592583), -INT32_C(  2101184466), -INT32_C(  1152826228),
         INT32_C(  1587676386),  INT32_C(  2074015086),  INT32_C(  1908069197), -INT32_C(   875208965),  INT32_C(  1610654360), -INT32_C(  1787212186),  INT32_C(  1394031814),  INT32_C(  1645109376) },
      {  INT64_C(                  30),  INT64_C(                  31) },
      {  INT32_C(           3),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           2),
         INT32_C(           0),  INT32_C(           1),  INT32_C(           1),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           1),  INT32_C(           1) } },
    { UINT16_C(63452),
      { -INT32_C(   666484129), -INT32_C(   283898665),  INT32_C(  1850709087), -INT32_C(  1239436042), -INT32_C(   719993465),  INT32_C(   484329144),  INT32_C(  1783972979), -INT32_C(  1570623165),
         INT32_C(  1266395252),  INT32_C(   339382196),  INT32_C(   444762660), -INT32_C(   959340226),  INT32_C(   513533542), -INT32_C(  1623557844),  INT32_C(    17469374),  INT32_C(   899968193) },
      {  INT64_C(                  26),  INT64_C(                  12) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(          27),  INT32_C(          45),  INT32_C(          53),  INT32_C(           0),  INT32_C(          26),  INT32_C(          40),
         INT32_C(          18),  INT32_C(           5),  INT32_C(           6),  INT32_C(           0),  INT32_C(           7),  INT32_C(          39),  INT32_C(           0),  INT32_C(          13) } },
    { UINT16_C(58788),
      { -INT32_C(  1117859709),  INT32_C(  2051873904), -INT32_C(  1025110498), -INT32_C(  1008600509), -INT32_C(  1619095614),  INT32_C(  1342030690), -INT32_C(  1878131385), -INT32_C(   428494494),
        -INT32_C(   240921471),  INT32_C(   241955056),  INT32_C(  1104171518),  INT32_C(   235254091), -INT32_C(  1548909759), -INT32_C(  1896699321),  INT32_C(   169803687),  INT32_C(   837850288) },
      {  INT64_C(                  16),  INT64_C(                   5) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(       49894),  INT32_C(           0),  INT32_C(           0),  INT32_C(       20477),  INT32_C(           0),  INT32_C(       58997),
         INT32_C(       61859),  INT32_C(           0),  INT32_C(       16848),  INT32_C(           0),  INT32_C(           0),  INT32_C(       36594),  INT32_C(        2590),  INT32_C(       12784) } },
    { UINT16_C(58956),
      { -INT32_C(  1080978483),  INT32_C(  1086208033),  INT32_C(   852782658), -INT32_C(  1027195745), -INT32_C(    78558572), -INT32_C(  1875693108), -INT32_C(  1772544932),  INT32_C(   326936134),
         INT32_C(  1540492601), -INT32_C(  2003070906), -INT32_C(  1648660482),  INT32_C(  1063289259), -INT32_C(  1757695541),  INT32_C(  1042837218), -INT32_C(   791379574),  INT32_C(  1642287399) },
      {  INT64_C(                   8),  INT64_C(                   6) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(     3331182),  INT32_C(    12764732),  INT32_C(           0),  INT32_C(           0),  INT32_C(     9853212),  INT32_C(           0),
         INT32_C(           0),  INT32_C(     8952720),  INT32_C(    10337135),  INT32_C(           0),  INT32_C(           0),  INT32_C(     4073582),  INT32_C(    13685889),  INT32_C(     6415185) } },
    { UINT16_C(60401),
      { -INT32_C(   212216885), -INT32_C(   428481774), -INT32_C(  1774740301), -INT32_C(  1186228483),  INT32_C(  1729138746), -INT32_C(  1269836077), -INT32_C(   118179769), -INT32_C(  1193023764),
        -INT32_C(   877969991), -INT32_C(   726523872), -INT32_C(  1167398467), -INT32_C(  1166756225),  INT32_C(   438404166), -INT32_C(  1563528869),  INT32_C(   379241001),  INT32_C(  1657700008) },
      {  INT64_C(                  10),  INT64_C(                   2) },
      {  INT32_C(     3987060),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(     1688612),  INT32_C(     2954229),  INT32_C(     4078894),  INT32_C(     3029241),
         INT32_C(     3336911),  INT32_C(     3484808),  INT32_C(           0),  INT32_C(     3054893),  INT32_C(           0),  INT32_C(     2667420),  INT32_C(      370352),  INT32_C(     1618847) } },
    { UINT16_C( 9226),
      { -INT32_C(    23763664), -INT32_C(  1564361209), -INT32_C(  1574934060), -INT32_C(   115549237),  INT32_C(  1725478582),  INT32_C(   511746317), -INT32_C(   324775702), -INT32_C(  1760514458),
         INT32_C(   446014739),  INT32_C(   951866980), -INT32_C(  1948525376),  INT32_C(  1854207927), -INT32_C(  1160487507), -INT32_C(   388475650),  INT32_C(   181763236),  INT32_C(  1034020138) },
      {  INT64_C(                   9),  INT64_C(                  20) },
      {  INT32_C(           0),  INT32_C(     5333215),  INT32_C(           0),  INT32_C(     8162925),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(     4582894),  INT32_C(           0),  INT32_C(           0),  INT32_C(     7629866),  INT32_C(           0),  INT32_C(           0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_srl_epi32(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_srl_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t a[8];
    const int64_t b[2];
    const int64_t r[8];
  } test_vec[] = {
    { {  INT64_C( 4503272731429987359), -INT64_C( 2015384591523464281),  INT64_C( 8370329603485121851),  INT64_C( 8741546637051844249),
         INT64_C( 7244483912508171930),  INT64_C( 2420618720360332110),  INT64_C( 4422947859294142848),  INT64_C( 3359849781269978573) },
      {  INT64_C(                  37),  INT64_C(                  38) },
      {  INT64_C(            32765621),  INT64_C(           119553875),  INT64_C(            60902163),  INT64_C(            63603122),
         INT64_C(            52710557),  INT64_C(            17612319),  INT64_C(            32181181),  INT64_C(            24446124) } },
    { { -INT64_C( 7637003490261708254),  INT64_C( 6415480880923232326),  INT64_C( 9218637000122400785),  INT64_C( 4196059733781788805),
        -INT64_C(  930270758595148872),  INT64_C( 4864966429638513740), -INT64_C( 4010865298925415083), -INT64_C( 2947988449335396157) },
      {  INT64_C(                  25),  INT64_C(                  63) },
      {  INT64_C(        322155373795),  INT64_C(        191196229485),  INT64_C(        274736791852),  INT64_C(        125052324944),
         INT64_C(        522031584832),  INT64_C(        144987297941),  INT64_C(        430222713195),  INT64_C(        461898911725) } },
    { { -INT64_C( 6686951679321549528), -INT64_C( 6351318751585002523), -INT64_C( 1572949627593083733), -INT64_C(  631318895338144874),
         INT64_C( 2734693502203073391), -INT64_C( 5541956386463059407),  INT64_C( 3082894904229679673), -INT64_C( 5609612383155050401) },
      {  INT64_C(                  19),  INT64_C(                  33) },
      {  INT64_C(      22430023945594),  INT64_C(      23070192951439),  INT64_C(      32184208767159),  INT64_C(      33980226856940),
         INT64_C(       5216013912588),  INT64_C(      24613929152005),  INT64_C(       5880155380687),  INT64_C(      24484885579213) } },
    { { -INT64_C( 7032322167997686871),  INT64_C( 3695752672797643348), -INT64_C( 6275681140897592453),  INT64_C( 2911922955017037974),
         INT64_C( 5599047430447746517), -INT64_C( 1275313147080310623),  INT64_C( 6872898518047554424), -INT64_C( 8520621702864799464) },
      {  INT64_C(                  37),  INT64_C(                  18) },
      {  INT64_C(            83050849),  INT64_C(            26890139),  INT64_C(            88556138),  INT64_C(            21187027),
         INT64_C(            40738431),  INT64_C(           124938603),  INT64_C(            50006918),  INT64_C(            72222045) } },
    { {  INT64_C( 4873285309233805945),  INT64_C(  439834950256813021), -INT64_C( 2716455584873858879),  INT64_C( 8498945592412536537),
         INT64_C( 3853490666578965515), -INT64_C( 5633192859071523523),  INT64_C( 6376453306728335769), -INT64_C(  565776880762709068) },
      {  INT64_C(                  29),  INT64_C(                  45) },
      {  INT64_C(          9077201242),  INT64_C(           819256436),  INT64_C(         29299945549),  INT64_C(         15830519781),
         INT64_C(          7177685697),  INT64_C(         23867099014),  INT64_C(         11877069821),  INT64_C(         33305896805) } },
    { {  INT64_C( 8157098361486837595),  INT64_C( 1321040308684797296),  INT64_C( 3012847868082094884),  INT64_C( 8923801435785072389),
         INT64_C( 8497317249283403709),  INT64_C( 8789380872681950910),  INT64_C( 2017793055357488554), -INT64_C( 8480534500030408781) },
      {  INT64_C(                  59),  INT64_C(                  24) },
      {  INT64_C(                  14),  INT64_C(                   2),  INT64_C(                   5),  INT64_C(                  15),
         INT64_C(                  14),  INT64_C(                  15),  INT64_C(                   3),  INT64_C(                  17) } },
    { { -INT64_C(  337988464786431205), -INT64_C( 1610283556321391280), -INT64_C( 6926550933078561340), -INT64_C( 7089296378098830461),
         INT64_C( 6893073757373453926),  INT64_C( 6645101486772353452), -INT64_C( 2301568806200316975),  INT64_C( 3098962276046799842) },
      {  INT64_C(                  38),  INT64_C(                  28) },
      {  INT64_C(            65879269),  INT64_C(            61250686),  INT64_C(            41910218),  INT64_C(            41318154),
         INT64_C(            25076856),  INT64_C(            24174738),  INT64_C(            58735805),  INT64_C(            11273959) } },
    { {  INT64_C( 5894349276315664526),  INT64_C( 3522860139222001906),  INT64_C( 4256409894813416583), -INT64_C( 5955799268209984695),
         INT64_C( 1855594266527086592), -INT64_C( 5539512011075848627),  INT64_C(  426499171783667312),  INT64_C( 8791384339317111350) },
      {  INT64_C(                  19),  INT64_C(                  44) },
      {  INT64_C(      11242579033500),  INT64_C(       6719322470134),  INT64_C(       8118457593561),  INT64_C(      23824586497305),
         INT64_C(       3539265187315),  INT64_C(      24618591428058),  INT64_C(        813482612197),  INT64_C(      16768234900125) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_srl_epi64(a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_srl_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t src[8];
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[2];
    const int64_t r[8];
  } test_vec[] = {
{ {  INT64_C( 5018051879103075663), -INT64_C(  660831149718276767), -INT64_C( 4444598874757254777), -INT64_C( 3636321904948482748),
         INT64_C(  694495148200162758), -INT64_C(  912842675246716212),  INT64_C( 3729394938595987999),  INT64_C( 5382103391663641445) },
      UINT8_C(  7),
      {  INT64_C( 8469222834209532005), -INT64_C( 8151281972419631149), -INT64_C(  875474453646156037), -INT64_C( 3146959030850175687),
        -INT64_C( 8202998051811360151), -INT64_C( 2609501873265243137),  INT64_C( 3431787955991037454),  INT64_C( 4695510415541147622) },
      {  INT64_C(                  52),  INT64_C(                  41) },
      {  INT64_C(                1880),  INT64_C(                2286),  INT64_C(                3901), -INT64_C( 3636321904948482748),
         INT64_C(  694495148200162758), -INT64_C(  912842675246716212),  INT64_C( 3729394938595987999),  INT64_C( 5382103391663641445) } },
    { {  INT64_C( 4774085380789252891),  INT64_C( 6351976018652674572),  INT64_C( 7285695253303204985),  INT64_C( 1704423325787105127),
        -INT64_C(  205924299156423802), -INT64_C(  863837376267240669),  INT64_C( 5459588585176950308),  INT64_C(  907179034420130690) },
      UINT8_C(237),
      { -INT64_C( 3509752084262096962), -INT64_C( 2384022715044792899),  INT64_C( 4709090717786589566),  INT64_C( 3258586805351995736),
        -INT64_C( 5862104578432690494), -INT64_C( 6022899097165107309), -INT64_C( 7856847056654750553),  INT64_C( 6004729387056741816) },
      {  INT64_C(                  54),  INT64_C(                  16) },
      {  INT64_C(                 829),  INT64_C( 6351976018652674572),  INT64_C(                 261),  INT64_C(                 180),
        -INT64_C(  205924299156423802),  INT64_C(                 689),  INT64_C(                 587),  INT64_C(                 333) } },
    { { -INT64_C( 5347199935727591939), -INT64_C( 1761696095209150550),  INT64_C( 4827803292971284187),  INT64_C( 3812233551889196554),
         INT64_C( 8473019998232436452), -INT64_C( 9115217864340507683),  INT64_C( 6537717375098455028), -INT64_C( 7821090437979903991) },
      UINT8_C(198),
      {  INT64_C( 9147944483022796001), -INT64_C( 8004967957813555785),  INT64_C(  357537112035189065), -INT64_C( 8293301012967246278),
         INT64_C( 3981911599859595706),  INT64_C( 6035127009660397092),  INT64_C( 1627267161321939317),  INT64_C( 5234344461575145580) },
      {  INT64_C(                   1),  INT64_C(                   2) },
      { -INT64_C( 5347199935727591939),  INT64_C( 5220888057947997915),  INT64_C(  178768556017594532),  INT64_C( 3812233551889196554),
         INT64_C( 8473019998232436452), -INT64_C( 9115217864340507683),  INT64_C(  813633580660969658),  INT64_C( 2617172230787572790) } },
    { { -INT64_C(  753457344752823400),  INT64_C( 9100845639399228278),  INT64_C( 6691762627973503226), -INT64_C( 6772517690425994392),
         INT64_C( 4611865007080578178), -INT64_C( 4987858906752217499), -INT64_C( 5481882081349639960), -INT64_C( 5102383525635684220) },
      UINT8_C( 77),
      { -INT64_C( 3220258701015792569), -INT64_C( 4606252918426593125), -INT64_C( 7433827182372543520), -INT64_C( 2636424837617130645),
         INT64_C( 8279327878949470286), -INT64_C( 6427567103991252564),  INT64_C(   35410411469199188),  INT64_C( 2639257638627200305) },
      {  INT64_C(                   9),  INT64_C(                  60) },
      {  INT64_C(   29739229243542498),  INT64_C( 9100845639399228278),  INT64_C(   21509603303392593),  INT64_C(   30879529757993009),
         INT64_C( 4611865007080578178), -INT64_C( 4987858906752217499),  INT64_C(      69160959900779), -INT64_C( 5102383525635684220) } },
    { { -INT64_C(  474472491459740714),  INT64_C( 1413390369529169826), -INT64_C( 8618690178674731836),  INT64_C( 7765024437675857456),
        -INT64_C( 7951878168928412957), -INT64_C( 8653482228769463353),  INT64_C( 5473607594897955195), -INT64_C( 6430896436868883521) },
      UINT8_C(119),
      {  INT64_C( 6529930248793313686),  INT64_C( 3757589078185112322),  INT64_C( 7156672293724263016), -INT64_C(  213133291419162954),
         INT64_C( 7813308203475955998), -INT64_C( 7533049731877532699),  INT64_C( 1637473394195219917), -INT64_C( 3854038203012358113) },
      {  INT64_C(                  43),  INT64_C(                   6) },
      {  INT64_C(              742367),  INT64_C(              427188),  INT64_C(              813619),  INT64_C( 7765024437675857456),
         INT64_C(              888270),  INT64_C(             1240743),  INT64_C(              186159), -INT64_C( 6430896436868883521) } },
    { {  INT64_C( 6890553886565778985),  INT64_C( 2558441506522614282), -INT64_C( 8775070714056974145), -INT64_C( 5393012673236621375),
        -INT64_C( 3633876295361783311), -INT64_C( 7040528235959716310),  INT64_C( 7969224737570684516),  INT64_C( 4740220371150316935) },
      UINT8_C( 62),
      {  INT64_C( 8796319542675600136), -INT64_C( 8882514063471855023), -INT64_C( 8572988487162016737), -INT64_C( 1585517887377303268),
         INT64_C( 2309471185300160654),  INT64_C( 9018229961860212253),  INT64_C( 1579359360142563581),  INT64_C( 7919253252302357531) },
      {  INT64_C(                  46),  INT64_C(                  26) },
      {  INT64_C( 6890553886565778985),  INT64_C(              135915),  INT64_C(              140314),  INT64_C(              239612),
         INT64_C(               32819),  INT64_C(              128156),  INT64_C( 7969224737570684516),  INT64_C( 4740220371150316935) } },
    { {  INT64_C( 7559772791110257371),  INT64_C( 7838035306306977657),  INT64_C( 4924922392629855747),  INT64_C( 1361601353282270225),
        -INT64_C( 2494766160983707733),  INT64_C( 1481962523173766543), -INT64_C( 8585818606364548669), -INT64_C(  429836139557950017) },
      UINT8_C(153),
      {  INT64_C( 1679238285005550319), -INT64_C( 6570405501343403503),  INT64_C( 3222506128052807406), -INT64_C( 6472670232636117335),
         INT64_C( 1499612929985882470),  INT64_C( 2435025945260914468),  INT64_C( 3730338983580460633),  INT64_C( 1936251741381476225) },
      {  INT64_C(                   6),  INT64_C(                  40) },
      {  INT64_C(   26238098203211723),  INT64_C( 7838035306306977657),  INT64_C( 4924922392629855747),  INT64_C(  187094903766772410),
         INT64_C(   23431452031029413),  INT64_C( 1481962523173766543), -INT64_C( 8585818606364548669),  INT64_C(   30253933459085566) } },
    { {  INT64_C( 6606217542241028080), -INT64_C( 9219868554947169474),  INT64_C( 1856557821955333813), -INT64_C(  210876376874155225),
        -INT64_C( 8426757936638374440), -INT64_C( 3856056457707543913),  INT64_C( 1369719826837412456), -INT64_C( 7978806546093840581) },
      UINT8_C(227),
      { -INT64_C(  903047239936490564),  INT64_C(  660572222557549369), -INT64_C(  236148617042914490),  INT64_C(  145317255498170763),
         INT64_C( 2940386493423826057), -INT64_C( 1241562307819219252),  INT64_C( 5636579028621965864), -INT64_C( 4867983057477439960) },
      {  INT64_C(                  30),  INT64_C(                  58) },
      {  INT64_C(         16338840903),  INT64_C(           615205823),  INT64_C( 1856557821955333813), -INT64_C(  210876376874155225),
        -INT64_C( 8426757936638374440),  INT64_C(         16023574178),  INT64_C(          5249473292),  INT64_C(         12646206669) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi64(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_srl_epi64(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_srl_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[2];
    const int64_t r[8];
  } test_vec[] = {
{ UINT8_C(  2),
      {  INT64_C( 1273292168187332866),  INT64_C( 7422134831920816881),  INT64_C( 8778785881423789008), -INT64_C( 6870649568514933397),
         INT64_C( 5972440929581448533),  INT64_C( 2069270216126665473),  INT64_C( 2304214308246073665),  INT64_C( 6084761119011074867) },
      {  INT64_C(                  33),  INT64_C(                  10) },
      {  INT64_C(                   0),  INT64_C(           864050215),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(190),
      {  INT64_C( 1715469743755078245),  INT64_C( 3883908348811583318), -INT64_C( 6110753254294327408), -INT64_C( 9052310990052255468),
        -INT64_C( 4962707970185112987),  INT64_C( 3589194774746904202), -INT64_C( 2682461174757816112),  INT64_C( 8697536180863588116) },
      {  INT64_C(                  17),  INT64_C(                  30) },
      {  INT64_C(                   0),  INT64_C(      29631869116299),  INT64_C(      94116140895196),  INT64_C(      71673836392649),
         INT64_C(     102875031307406),  INT64_C(      27383382986045),  INT64_C(                   0),  INT64_C(      66356934973629) } },
    { UINT8_C( 54),
      { -INT64_C( 5799555120303860877),  INT64_C( 4096933604670747731),  INT64_C(  231998857741135549), -INT64_C( 3315171190985936359),
         INT64_C( 2720388382403328952),  INT64_C( 8797552889633669751),  INT64_C( 6835358425641782045),  INT64_C( 7418312242013505733) },
      {  INT64_C(                  27),  INT64_C(                  45) },
      {  INT64_C(                   0),  INT64_C(         30524534021),  INT64_C(          1728526187),  INT64_C(                   0),
         INT64_C(         20268472898),  INT64_C(         65546876859),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(110),
      { -INT64_C(  890849150588711221), -INT64_C( 1407084915905673929), -INT64_C( 1553937031705302402), -INT64_C( 6671174823202003303),
         INT64_C( 7432386549469224648),  INT64_C( 5516411252743718633),  INT64_C( 6273357867429032100), -INT64_C( 7723181002292161705) },
      {  INT64_C(                  24),  INT64_C(                   5) },
      {  INT64_C(                   0),  INT64_C(       1015642831194),  INT64_C(       1006889762997),  INT64_C(        701878622204),
         INT64_C(                   0),  INT64_C(        328803733154),  INT64_C(        373921267237),  INT64_C(                   0) } },
    { UINT8_C(  1),
      { -INT64_C( 2551546160007398705), -INT64_C( 1583229024549091734), -INT64_C( 4637370017592110076), -INT64_C( 4443569649745975869),
        -INT64_C( 7663831157342858114), -INT64_C( 8078959051578389152), -INT64_C( 4241788306351701286),  INT64_C( 1512768116375507273) },
      {  INT64_C(                  62),  INT64_C(                  40) },
      {  INT64_C(                   3),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C( 21),
      {  INT64_C( 8563556369835754345),  INT64_C( 1362375245873026039), -INT64_C( 2932503238363277827), -INT64_C(  467743725081221479),
         INT64_C( 7988363705945922112),  INT64_C(  546549678847879282), -INT64_C( 9007248830868667823),  INT64_C( 1945960595755801977) },
      {  INT64_C(                  22),  INT64_C(                  52) },
      {  INT64_C(       2041710941752),  INT64_C(                   0),  INT64_C(       3698883255802),  INT64_C(                   0),
         INT64_C(       1904574324118),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(180),
      { -INT64_C( 7627373761685227045), -INT64_C( 5077655539464261875), -INT64_C( 4634773583306946036),  INT64_C( 3241203397608668373),
        -INT64_C( 4208050115763857397), -INT64_C( 6766153118426920401), -INT64_C( 8678233403792115242), -INT64_C( 1354797465776172876) },
      {  INT64_C(                  15),  INT64_C(                  49) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(     421507888501056),  INT64_C(                   0),
         INT64_C(     434530455259573),  INT64_C(     356463347024006),  INT64_C(                   0),  INT64_C(     521604815915935) } },
    { UINT8_C(110),
      { -INT64_C(  701251862189271592), -INT64_C( 8672589849232389770), -INT64_C( 4551220995287630154), -INT64_C( 6372748713757113748),
        -INT64_C( 3664950061693423865),  INT64_C( 8559287630595816389), -INT64_C( 7435569239400424336), -INT64_C(  646749299245744753) },
      {  INT64_C(                  31),  INT64_C(                  18) },
      {  INT64_C(                   0),  INT64_C(          4551445238),  INT64_C(          6470607164),  INT64_C(          5622392222),
         INT64_C(                   0),  INT64_C(          3985728896),  INT64_C(          5127477848),  INT64_C(                   0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_srl_epi64(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_srl_epi16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_srl_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_srl_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_srl_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_srl_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_srl_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_srl_epi64)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
